<?php
/* --------------------------------------------------------------
   WithdrawalMapper.php 2020-01-23
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\Repository;

use Gambio\Admin\Withdrawal\Interfaces\WithdrawalIds;
use Gambio\Admin\Withdrawal\Interfaces\Withdrawal;
use Gambio\Admin\Withdrawal\Models\WithdrawalId;
use Gambio\Admin\Withdrawal\Models\WithdrawalIds as WithdrawalIdsModel;
use Gambio\Admin\Withdrawal\WithdrawalFactory;

/**
 * Class WithdrawalMapper
 *
 * @package Gambio\Admin\Withdrawal\Repository
 */
class WithdrawalMapper
{
    /**
     * @var WithdrawalFactory
     */
    private $factory;
    
    
    /**
     * WithdrawalMapper constructor.
     *
     * @param WithdrawalFactory $factory
     */
    public function __construct(WithdrawalFactory $factory)
    {
        $this->factory = $factory;
    }
    
    
    /**
     * Maps database data structure to internal domain data structure.
     *
     * @param array $data
     *
     * @return Withdrawal
     */
    public function mapWithdrawal(array $data): Withdrawal
    {
        $orderId = ($data['order_id'] !== null) ? (int)$data['order_id'] : $data['order_id'];
        $order   = $this->factory->createOrderDetails($orderId,
                                                      $data['order_date'],
                                                      $data['delivery_date']);
        
        $address = $this->factory->createCustomerAddress($data['customer_street_address'],
                                                         $data['customer_postcode'],
                                                         $data['customer_city'],
                                                         $data['customer_country']);
        
        $customerId = ($data['customer_id'] !== null) ? (int)$data['customer_id'] : $data['customer_id'];
        $customer   = $this->factory->createCustomerDetails($data['customer_gender'],
                                                            $data['customer_firstname'],
                                                            $data['customer_lastname'],
                                                            $address,
                                                            $data['customer_email'],
                                                            $customerId);
        
        return $this->factory->createWithdrawal($order,
                                                $customer,
                                                $data['withdrawal_date'],
                                                $data['withdrawal_content'],
                                                (bool)$data['created_by_admin'],
                                                (int)$data['withdrawal_id'],
                                                $data['date_created']);
    }
    
    
    /**
     * Maps list of IDs to a internal IDs object.
     *
     * @param int[] $ids
     *
     * @return WithdrawalIds
     */
    public function mapWithdrawalIds(array $ids): WithdrawalIds
    {
        $collection = [];
        foreach ($ids as $reference => $id) {
            $collection[$reference] = WithdrawalId::create($id);
        }
        
        return WithdrawalIdsModel::createWithReferences($collection);
    }
}